<?php
session_start();
require_once '../config/db.php';
require_once '../auth/auth.php';
require_once '../includes/auth_check.php';
require_once '../helpers/helpers.php';
require_once '../helpers/favicon_helper.php';

// Check if user is logged in
if (!isset($_SESSION['user_id'])) {
    header('Location: ../auth/login.php');
    exit();
}

$activePage = 'profile';
$success = '';
$error = '';

// Get user data
try {
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$_SESSION['user_id']]);
    $user = $stmt->fetch();
} catch (PDOException $e) {
    $error = 'Error fetching user data';
}

// Handle profile update
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    if (isset($_POST['action'])) {
        try {
            switch ($_POST['action']) {
                case 'update_profile':
                    $stmt = $pdo->prepare("UPDATE users SET username = ?, email = ? WHERE id = ?");
                    $stmt->execute([
                        $_POST['username'],
                        $_POST['email'],
                        $_SESSION['user_id']
                    ]);
                    $success = 'Profile updated successfully';
                    break;

                case 'update_password':
                    if (password_verify($_POST['current_password'], $user['password'])) {
                        if ($_POST['new_password'] === $_POST['confirm_password']) {
                            $stmt = $pdo->prepare("UPDATE users SET password = ? WHERE id = ?");
                            $stmt->execute([
                                password_hash($_POST['new_password'], PASSWORD_DEFAULT),
                                $_SESSION['user_id']
                            ]);
                            $success = 'Password updated successfully';
                        } else {
                            $error = 'New passwords do not match';
                        }
                    } else {
                        $error = 'Current password is incorrect';
                    }
                    break;

                case 'upload_picture':
                    if (isset($_FILES['profile_picture']) && $_FILES['profile_picture']['error'] === 0) {
                        $allowed = ['jpg', 'jpeg', 'png', 'gif'];
                        $filename = $_FILES['profile_picture']['name'];
                        $ext = strtolower(pathinfo($filename, PATHINFO_EXTENSION));
                        
                        if (in_array($ext, $allowed)) {
                            $newFilename = 'profile_' . $_SESSION['user_id'] . '.' . $ext;
                            $uploadPath = '../uploads/profiles/' . $newFilename;
                            
                            if (!is_dir('../uploads/profiles')) {
                                mkdir('../uploads/profiles', 0777, true);
                            }
                            
                            if (move_uploaded_file($_FILES['profile_picture']['tmp_name'], $uploadPath)) {
                                $stmt = $pdo->prepare("UPDATE users SET profile_picture = ? WHERE id = ?");
                                $stmt->execute([$newFilename, $_SESSION['user_id']]);
                                
                                // Refresh user data to show updated profile picture
                                $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
                                $stmt->execute([$_SESSION['user_id']]);
                                $user = $stmt->fetch();
                                
                                $success = 'Profile picture updated successfully';
                            } else {
                                $error = 'Error uploading file';
                            }
                        } else {
                            $error = 'Invalid file type. Allowed types: ' . implode(', ', $allowed);
                        }
                    }
                    break;
            }
        } catch (PDOException $e) {
            $error = 'Database error: ' . $e->getMessage();
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Profile - POS System</title>
    <?php echo getFaviconLink($pdo); ?>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;600;700&display=swap" rel="stylesheet">
    <style>
        body { background: linear-gradient(135deg, #f8fafc 0%, #e0e7ff 100%); font-family: 'Inter', sans-serif; }
        .card, .profile-card { border-radius: 16px; box-shadow: 0 2px 12px rgba(37,99,235,0.06); border: none; }
        .card-header { background: none; border-bottom: none; }
        .btn-primary, .btn-warning, .btn-info, .btn-danger { border-radius: 8px; font-weight: 600; }
        .form-label { font-weight: 600; color: #2563eb; }
        .modal-title { color: #2563eb; font-weight: 700; }
        .summary-title { font-size: 1.1rem; color: #64748b; font-weight: 600; }
        .summary-value { font-size: 1.7rem; font-weight: 700; color: #222; }
        .section-header { font-size: 1.3rem; font-weight: 700; color: #2563eb; margin-bottom: 1.5rem; display: flex; align-items: center; gap: 0.5rem; }
        .profile-card {
            max-width: 800px;
            margin: 40px auto;
            padding: 2rem;
        }
        .profile-header {
            text-align: center;
            margin-bottom: 2rem;
        }
        .profile-picture {
            width: 150px;
            height: 150px;
            border-radius: 50%;
            margin: 0 auto 1rem;
            background: #2563eb;
            display: flex;
            align-items: center;
            justify-content: center;
            color: white;
            font-size: 3rem;
            font-weight: 700;
            position: relative;
            overflow: hidden;
        }
        .profile-picture img {
            width: 100%;
            height: 100%;
            object-fit: cover;
        }
        .profile-picture-upload {
            position: absolute;
            bottom: 0;
            left: 0;
            right: 0;
            background: rgba(0,0,0,0.5);
            color: white;
            padding: 0.5rem;
            text-align: center;
            cursor: pointer;
            opacity: 0;
            transition: opacity 0.2s;
        }
        .profile-picture:hover .profile-picture-upload {
            opacity: 1;
        }
        .profile-name {
            font-size: 1.5rem;
            font-weight: 700;
            margin-bottom: 0.5rem;
        }
        .profile-email {
            color: #666;
        }
        .nav-pills .nav-link {
            color: #444;
            border-radius: 8px;
            padding: 0.8rem 1.5rem;
        }
        .nav-pills .nav-link.active {
            background: #2563eb;
            color: white;
        }
        .tab-content {
            padding: 1.5rem 0;
        }
    </style>
</head>
<body>
    <div class="container-fluid py-4">
        <div class="row">
            <?php include '../includes/sidebar.php'; ?>
            <!-- Main Content -->
            <div class="col-md-10 ms-sm-auto px-md-4">
                <?php include '../includes/topbar.php'; ?>
                
                <div class="profile-card">
                    <?php if ($success): ?>
                        <div class="alert alert-success"><?php echo $success; ?></div>
                    <?php endif; ?>
                    <?php if ($error): ?>
                        <div class="alert alert-danger"><?php echo $error; ?></div>
                    <?php endif; ?>

                    <div class="profile-header">
                        <div class="profile-picture">
                            <?php if (!empty($user['profile_picture'])): ?>
                                <img src="../uploads/profiles/<?php echo htmlspecialchars($user['profile_picture']); ?>" alt="Profile Picture">
                            <?php else: ?>
                                <?php echo strtoupper(substr($user['username'], 0, 1)); ?>
                            <?php endif; ?>
                            <label class="profile-picture-upload" for="profilePictureInput">
                                <i class="fas fa-camera"></i> Change Picture
                            </label>
                        </div>
                        <div class="profile-name"><?php echo htmlspecialchars($user['username']); ?></div>
                        <div class="profile-email"><?php echo htmlspecialchars($user['email']); ?></div>
                    </div>

                    <ul class="nav nav-pills mb-4" id="profileTabs" role="tablist">
                        <li class="nav-item" role="presentation">
                            <button class="nav-link active" data-bs-toggle="pill" data-bs-target="#profile" type="button">Profile</button>
                        </li>
                        <li class="nav-item" role="presentation">
                            <button class="nav-link" data-bs-toggle="pill" data-bs-target="#password" type="button">Password</button>
                        </li>
                    </ul>

                    <div class="tab-content">
                        <div class="tab-pane fade show active" id="profile">
                            <form method="POST" class="needs-validation" novalidate>
                                <input type="hidden" name="action" value="update_profile">
                                <div class="mb-3">
                                    <label class="form-label">Username</label>
                                    <input type="text" class="form-control" name="username" value="<?php echo htmlspecialchars($user['username']); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Email</label>
                                    <input type="email" class="form-control" name="email" value="<?php echo htmlspecialchars($user['email']); ?>" required>
                                </div>
                                <button type="submit" class="btn btn-primary">Update Profile</button>
                            </form>
                        </div>

                        <div class="tab-pane fade" id="password">
                            <form method="POST" class="needs-validation" novalidate>
                                <input type="hidden" name="action" value="update_password">
                                <div class="mb-3">
                                    <label class="form-label">Current Password</label>
                                    <input type="password" class="form-control" name="current_password" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">New Password</label>
                                    <input type="password" class="form-control" name="new_password" required>
                                </div>
                                <div class="mb-3">
                                    <label class="form-label">Confirm New Password</label>
                                    <input type="password" class="form-control" name="confirm_password" required>
                                </div>
                                <button type="submit" class="btn btn-primary">Update Password</button>
                            </form>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <form id="profilePictureForm" method="POST" enctype="multipart/form-data" style="display: none;">
        <input type="hidden" name="action" value="upload_picture">
        <input type="file" id="profilePictureInput" name="profile_picture" accept="image/*">
    </form>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Handle profile picture upload
        document.getElementById('profilePictureInput').addEventListener('change', function() {
            if (this.files && this.files[0]) {
                document.getElementById('profilePictureForm').submit();
            }
        });

        // Form validation
        (function() {
            'use strict';
            var forms = document.querySelectorAll('.needs-validation');
            Array.prototype.slice.call(forms).forEach(function(form) {
                form.addEventListener('submit', function(event) {
                    if (!form.checkValidity()) {
                        event.preventDefault();
                        event.stopPropagation();
                    }
                    form.classList.add('was-validated');
                }, false);
            });
        })();
    </script>
</body>
</html> 